<?php
namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use App\Models\QueueItem;
use App\Models\Campaign;
use App\Models\CampaignStat;
use App\Models\Endpoint;
use App\Models\Setting;

class CampaignSend extends Command
{
    protected $signature = 'campaign:send';
    protected $description = 'Process queued campaign items and send to endpoint';

    public function handle()
    {
        $batch = (int) (Setting::get('worker_batch_size', 20));
        $sleep = (int) (Setting::get('worker_sleep_between', 1));
        $maxAttempts = (int) (Setting::get('worker_max_attempts', 3));

        $items = QueueItem::where('status', 'pending')->orderBy('id')->limit($batch)->get();
        foreach ($items as $it) {
            // Lock item
            $updated = QueueItem::where('id', $it->id)->where('status','pending')->update([
                'status'=>'processing', 'attempts'=>DB::raw('attempts+1')
            ]);
            if (!$updated) continue;

            $payload = json_decode($it->payload_json, true) ?? [];

            $campaign = Campaign::find($it->campaign_id);
            if (!$campaign) { $it->status='failed'; $it->last_error='Campaign not found'; $it->save(); continue; }
            $endpoint = Endpoint::find($campaign->endpoint_id);
            if (!$endpoint || !$endpoint->is_active) { $it->status='failed'; $it->last_error='Endpoint inactive'; $it->save(); continue; }

            $res = $this->sendRequest($endpoint->method, $endpoint->url, $payload);
            $success = $this->isSuccess($res['code'], $res['body']);

            $it->status = $success ? 'success' : (($it->attempts >= $maxAttempts) ? 'failed' : 'pending');
            $it->response_code = $res['code'];
            $it->response_body = $res['body'];
            $it->last_error = $res['error'];
            $it->save();

            // update stats
            if ($it->status === 'success') {
                CampaignStat::updateOrCreate(['campaign_id'=>$it->campaign_id], [])->increment('sent_success');
            } elseif ($it->status === 'failed') {
                CampaignStat::updateOrCreate(['campaign_id'=>$it->campaign_id], [])->increment('sent_failed');
            }

            sleep($sleep);
        }

        // Update campaign status
        $this->updateCampaignStatuses();

        // Cleanup old logs
        $retention = (int) (Setting::get('log_retention_days', 10));
        DB::statement("DELETE FROM queue_items WHERE created_at < DATE_SUB(NOW(), INTERVAL {$retention} DAY)");
    }

    protected function sendRequest(string $method, string $url, array $payload): array
    {
        try {
            if ($method === 'GET') {
                $url = $url.'?'.http_build_query($payload);
                $ch = curl_init($url);
            } else {
                $ch = curl_init($url);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
            }
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            $body = curl_exec($ch);
            $err = curl_error($ch);
            $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            return ['code'=>$code?:0, 'body'=>$body?:'', 'error'=>$err?:null];
        } catch (\Throwable $e) {
            return ['code'=>0,'body'=>'','error'=>$e->getMessage()];
        }
    }

    protected function isSuccess(int $code, string $body): bool
    {
        if ($code >= 200 && $code < 300) {
            $j = json_decode($body, true);
            return isset($j['status']) && (string)$j['status'] === '1';
        }
        return false;
    }

    protected function updateCampaignStatuses(): void
    {
        // sending if there is pending/processing; error if any failed and none pending; else done
        $campaignIds = Campaign::pluck('id')->all();
        foreach ($campaignIds as $cid) {
            $pending = QueueItem::where('campaign_id',$cid)->whereIn('status',['pending','processing'])->exists();
            $failed  = QueueItem::where('campaign_id',$cid)->where('status','failed')->exists();
            if ($pending) {
                Campaign::where('id',$cid)->update(['status'=>'sending']);
            } else {
                Campaign::where('id',$cid)->update(['status'=> $failed ? 'error' : 'done']);
            }
        }
    }
}
